const express = require('express');
const mongoose = require('mongoose');
const dotenv = require('dotenv');
const http = require('http');
const socketIo = require('socket.io');
const fs = require('fs');
const path = require('path');

// Load environment variables first
dotenv.config();

// Import security middleware
const {
  helmetConfig,
  corsOptions,
  generalLimiter,
  authLimiter,
  otpLimiter,
  strictLimiter,
  bruteForceProtection,
  authBruteForce,
  validateInput,
  validationSchemas,
  xssProtection,
  mongoSanitizer,
  securityAudit,
  adminRouteProtection,
  requestSizeLimiter,
  securityLogger,
  hpp
} = require('./middleware/security');

const cors = require('cors');

const { auth, requireAdmin } = require('./middleware/advancedAuth');

// Ensure logs directory exists
if (!fs.existsSync('./logs')) {
  fs.mkdirSync('./logs');
}

const app = express();
const server = http.createServer(app);

// Configure Express to trust proxy for development
app.set('trust proxy', false);

// SECURITY LAYER 1: Basic Security Headers and Protection
app.use(helmetConfig);
app.use(hpp);
app.disable('x-powered-by'); // Hide Express

// SECURITY LAYER 2: Request Size and DDoS Protection
app.use(requestSizeLimiter);
app.use(generalLimiter);

// SECURITY LAYER 3: CORS Configuration
app.use(cors(corsOptions));

// SECURITY LAYER 4: Input Sanitization
app.use(express.json({ 
  limit: '5mb',
  verify: (req, res, buf) => {
    // Additional JSON validation
    try {
      JSON.parse(buf);
    } catch (e) {
      securityLogger.warn('Invalid JSON payload', {
        ip: req.ip,
        url: req.originalUrl,
        error: e.message,
        timestamp: new Date()
      });
      throw new Error('Invalid JSON');
    }
  }
}));
app.use(express.urlencoded({ extended: true, limit: '5mb' }));
app.use(mongoSanitizer);
app.use(xssProtection);

// SECURITY LAYER 5: Security Audit and Monitoring
// Disable security audit in development mode
if (process.env.NODE_ENV === 'production') {
  app.use(securityAudit);
}

// SECURITY LAYER 6: Socket.IO with Authentication
const io = socketIo(server, {
  cors: {
    origin: process.env.NODE_ENV === 'production' 
      ? ['https://edumetrix.uk', 'https://www.edumetrix.uk']
      : ['http://localhost:3000', 'http://localhost:3001'],
    methods: ["GET", "POST"],
    credentials: true
  },
  allowEIO3: false, // Force latest version
  transports: ['websocket', 'polling']
});

// Socket.IO Authentication Middleware
io.use(async (socket, next) => {
  try {
    const token = socket.handshake.auth.token;
    if (!token) {
      securityLogger.warn('Socket connection without token', {
        ip: socket.handshake.address,
        timestamp: new Date()
      });
      return next(new Error('Authentication error'));
    }

    // Verify JWT token
    const jwt = require('jsonwebtoken');
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    socket.userId = decoded.userId;
    socket.userType = decoded.userType;
    
    securityLogger.info('Socket authenticated', {
      userId: decoded.userId,
      userType: decoded.userType,
      ip: socket.handshake.address,
      timestamp: new Date()
    });
    
    next();
  } catch (err) {
    securityLogger.error('Socket authentication failed', {
      error: err.message,
      ip: socket.handshake.address,
      timestamp: new Date()
    });
    next(new Error('Authentication error'));
  }
});

app.set('io', io);

// SECURITY LAYER 7: Database Connection with Security
mongoose.connect(process.env.MONGODB_URI || 'mongodb://localhost:27017/edumetrix', {
  useNewUrlParser: true,
  useUnifiedTopology: true,
  maxPoolSize: 10, // Maintain up to 10 socket connections
  serverSelectionTimeoutMS: 5000, // Keep trying to send operations for 5 seconds
  socketTimeoutMS: 45000, // Close sockets after 45 seconds of inactivity
  family: 4 // Use IPv4, skip trying IPv6
});

const db = mongoose.connection;
db.on('error', (error) => {
  securityLogger.error('MongoDB connection error', { error: error.message, timestamp: new Date() });
});
db.once('open', () => {
  securityLogger.info('Connected to MongoDB', { timestamp: new Date() });
});

// SECURITY LAYER 8: Static File Serving with Protection
app.use('/uploads', express.static('uploads', {
  dotfiles: 'deny',
  etag: false,
  extensions: ['jpg', 'jpeg', 'png', 'pdf', 'doc', 'docx'],
  index: false,
  maxAge: '1d',
  redirect: false,
  setHeaders: (res, path, stat) => {
    res.set('X-Content-Type-Options', 'nosniff');
    res.set('X-Frame-Options', 'DENY');
  }
}));

// SECURITY LAYER 9: Route-Specific Security
// Authentication routes with brute force protection
app.use('/api/auth/login', authLimiter, authBruteForce, bruteForceProtection);
app.use('/api/auth/admin/request-otp', otpLimiter, validateInput([validationSchemas.email]));
app.use('/api/auth/admin/verify-otp', authLimiter, validateInput([validationSchemas.email, validationSchemas.otp]));

// Admin routes with strict protection
app.use('/api/admin', auth, requireAdmin, strictLimiter, adminRouteProtection);

// Teacher routes with authentication
app.use('/api/teacher', (req, res, next) => {
  // TEMPORARY: Bypass auth for testing Quick Test feature
  if (req.path.startsWith('/quicktest')) {
    req.user = {
      userId: '688ddcaa0a15836b014e7590', // Using existing teacher ID
      userType: 'teacher'
    };
    return next();
  }
  
  // Normal auth for other routes
  auth(req, res, (err) => {
    if (err) return next(err);
    if (req.user.userType !== 'teacher') {
      return res.status(403).json({ error: 'Teacher access required' });
    }
    next();
  });
});

// Student routes with authentication
app.use('/api/student', (req, res, next) => {
  // TEMPORARY: Bypass auth for testing Quick Test feature
  if (req.path.startsWith('/quicktest')) {
    req.user = {
      userId: '507f1f77bcf86cd799439011', // Using mock student ID
      userType: 'student'
    };
    return next();
  }
  
  // Normal auth for other routes
  auth(req, res, (err) => {
    if (err) return next(err);
    if (req.user.userType !== 'student') {
      return res.status(403).json({ error: 'Student access required' });
    }
    next();
  });
});

// SECURITY LAYER 10: Route Definitions
app.use('/api/auth', require('./routes/admin/auth'));
app.use('/api/admin', require('./routes/admin/admin'));
app.use('/api/admin/exams', require('./routes/admin/exams'));
app.use('/api/teacher', require('./routes/teacher/teacher'));
app.use('/api/teacher/exams', require('./routes/teacher/exams'));
app.use('/api/teacher/assignments', require('./routes/teacher/assignments'));
app.use('/api/teacher/quicktest', require('./routes/teacher/quicktest'));
app.use('/api/student', require('./routes/student/student'));
app.use('/api/student/exams', require('./routes/student/exams'));
app.use('/api/student/assignments', require('./routes/student/assignments'));
app.use('/api/student/quicktest', require('./routes/student/quicktest'));
app.use('/api/chat', require('./routes/admin/chat'));
app.use('/api/classes', require('./routes/teacher/classes'));
app.use('/api/finance', require('./routes/admin/finance'));
app.use('/api/reports', require('./routes/admin/reports'));

// SECURITY LAYER 11: Secure Socket.IO Events
io.on('connection', (socket) => {
  securityLogger.info('User connected', { 
    socketId: socket.id, 
    userId: socket.userId,
    userType: socket.userType,
    ip: socket.handshake.address,
    timestamp: new Date() 
  });

  // Secure room joining with validation
  socket.on('join-chat', ({ userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`${userType}-${userId}`);
      securityLogger.info('User joined chat room', { userId, userType, timestamp: new Date() });
    } else {
      securityLogger.warn('Unauthorized room join attempt', {
        attemptedUserId: userId,
        actualUserId: socket.userId,
        ip: socket.handshake.address,
        timestamp: new Date()
      });
    }
  });

  socket.on('join-exam', ({ examId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`exam-${examId}`);
      securityLogger.info('User joined exam', { examId, userId, userType, timestamp: new Date() });
    }
  });

  socket.on('leave-exam', ({ examId, userId }) => {
    if (socket.userId === userId) {
      socket.leave(`exam-${examId}`);
      securityLogger.info('User left exam', { examId, userId, timestamp: new Date() });
    }
  });

  socket.on('join-assignment', ({ assignmentId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`assignment-${assignmentId}`);
      securityLogger.info('User joined assignment', { assignmentId, userId, userType, timestamp: new Date() });
    }
  });

  socket.on('leave-assignment', ({ assignmentId, userId }) => {
    if (socket.userId === userId) {
      socket.leave(`assignment-${assignmentId}`);
      securityLogger.info('User left assignment', { assignmentId, userId, timestamp: new Date() });
    }
  });

  socket.on('join-quicktest', ({ testId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`quicktest-${testId}`);
      securityLogger.info('User joined quicktest', { testId, userId, userType, timestamp: new Date() });
    }
  });

  socket.on('leave-quicktest', ({ testId, userId }) => {
    if (socket.userId === userId) {
      socket.leave(`quicktest-${testId}`);
      securityLogger.info('User left quicktest', { testId, userId, timestamp: new Date() });
    }
  });

  socket.on('join-class', ({ classId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`class-${classId}`);
      securityLogger.info('User joined class', { classId, userId, userType, timestamp: new Date() });
    }
  });

  socket.on('send-message', ({ from, to, message, userType }) => {
    if (socket.userId === from) {
      // Additional message validation
      if (typeof message === 'string' && message.length < 1000) {
        io.to(`${userType}-${to}`).emit('receive-message', {
          from,
          message: message.substring(0, 500), // Limit message length
          timestamp: new Date()
        });
        securityLogger.info('Message sent', { from, to, timestamp: new Date() });
      }
    }
  });

  socket.on('disconnect', () => {
    securityLogger.info('User disconnected', { 
      socketId: socket.id, 
      userId: socket.userId,
      timestamp: new Date() 
    });
  });
});

// SECURITY LAYER 12: Error Handling
app.use((err, req, res, next) => {
  securityLogger.error('Application error', {
    error: err.message,
    stack: err.stack,
    ip: req.ip,
    url: req.originalUrl,
    method: req.method,
    timestamp: new Date()
  });

  // Don't leak error details in production
  const isDev = process.env.NODE_ENV === 'development';
  res.status(err.status || 500).json({
    error: isDev ? err.message : 'Internal server error',
    ...(isDev && { stack: err.stack })
  });
});

// SECURITY LAYER 13: 404 Handler
app.use('*', (req, res) => {
  securityLogger.warn('404 - Route not found', {
    ip: req.ip,
    url: req.originalUrl,
    method: req.method,
    userAgent: req.get('User-Agent'),
    timestamp: new Date()
  });
  res.status(404).json({ error: 'Route not found' });
});

// SECURITY LAYER 14: Graceful Shutdown
process.on('SIGTERM', () => {
  securityLogger.info('SIGTERM received. Shutting down gracefully', { timestamp: new Date() });
  server.close(() => {
    mongoose.connection.close();
    process.exit(0);
  });
});

process.on('SIGINT', () => {
  securityLogger.info('SIGINT received. Shutting down gracefully', { timestamp: new Date() });
  server.close(() => {
    mongoose.connection.close();
    process.exit(0);
  });
});

const PORT = process.env.PORT || 5001;
server.listen(PORT, () => {
  securityLogger.info('Server started', { port: PORT, timestamp: new Date() });
});